
#include "i2c.h"

#define UARTBASE 0xE000C000

#define RBR 0x00
#define THR 0x00
#define DLL 0x00
#define DLM 0x04
#define IER 0x04
#define IIR 0x08
#define FCR 0x08

#define LCR 0x0c
#define LSR 0x14
#define SCR 0x1c
#define ACR 0x20
#define FDR 0x28
#define TER 0x30

#define UREG(x) (((volatile unsigned char *)UARTBASE)[x])

#define U0THRE ((UREG(LSR) & (1<<5))) /* UART0 transmitter holding register is empty */
#define U0DR ((UREG(LSR) & (1<<0))) /* UART0 data ready */


#define I2CBASE  0xE001C000

#define I2CONSET 0x00
#define I2STAT   0x04
#define I2DAT    0x08
#define I2ADR    0x0c
#define I2SCLH   0x10
#define I2SCLL   0x14
#define I2CONCLR 0x18

#define IREG(x) (((volatile unsigned char *)I2CBASE)[x])


#define PINSEL0 (*((volatile unsigned char *) 0xE002C000))

void init_uart(void)
{
	UREG(FDR) = 0x10; /* DivAddVal = 0, MulVal = 1 */

	UREG(LCR) = 0x80;
	UREG(DLM) = 0x00;
	UREG(DLL) = 0x08; /* 14745600 / (16*115200) */
	UREG(LCR) = 0x13;
	UREG(FCR) = 0x07;
}

void init_pins(void)
{
	PINSEL0 = 0x00000055; /* P0.0 and P0.1 assigned to UART */
			      /* P0.2 and P0.3 assigned to I2C  */
}

void putch(char c) {
	while (!U0THRE);
	UREG(THR) = c;
}

void putstr(char *s) {
	while (*s) putch(*s++);
}

void putint(unsigned int n) {
	char s[11];
	int i;

	i = 10;
	s[i] = '\0';

	do {
		s[--i] = n % 10 + '0';
	} while ((n /= 10) > 0);

	putstr(s+i);
}

void puthex(unsigned int n) {
	char s[9];
	int i;

	i = 8;
	s[i] = '\0';

	do {
		int x = n % 16;
		if (x > 9)
			x += 'A' - '0' - 10;
		s[--i] = x + '0';
	} while ((n /= 16) > 0);

	putstr(s+i);
}

char getch(void) {
	while (!U0DR);
	return UREG(RBR);
}

void reply(char *str)
{
	putstr(str);
	putstr("\r\n");
}

unsigned int count = 0;

int main(void) {
	init_uart();
	init_i2c();
	init_pins();
	putstr("Your entire life has been a mathematical error... a mathematical error I'm about to correct!\r\n");

	while (1) {
		char c;
		putstr("prompt> ");
		c = getch();
		if (c == 0x0a)
			continue;
		putch(c);
		putstr("\r\n");
		switch (c & 0xdf) {
		case 0x0a:
		case 0x0d:
			break;
		case 'A':
			reply("apple");
			break;
		case 'C':
			count++;
			putstr("The current count is ");
			putint(count);
			reply(".");
			break;
		case 'H':
		case '?':
			reply("Help is not available. Try a psychiatrist.");
			break;
		case 'R':
			putstr("I2C register is: ");
			puthex(i2c_conreg());
			reply(".");
			break;
		case 'T':
			putstr("I2C status was: ");
			puthex(i2cstat);
			putstr(" I2C status is: ");
			puthex(i2c_statreg());
			reply(".");
			break;
		case 'S':
			putstr("Sending START... ");
			if (i2c_send_start())
				reply("OK");
			else
				reply("FAIL");
			break;
		case 'D':
			putstr("Sending address... ");
			if (i2c_send_address(0x53, TRUE))
				reply("OK");
			else
				reply("FAIL");
			break;
		case 'B':
			putstr("Sending bytes... ");
			if (i2c_send_data(0xfe))
				reply("OK");
			else
				reply("FAIL");

			if (i2c_send_data(0x04))
				reply("OK");
			else
				reply("FAIL");
			break;
		case 'O':
			putstr("Sending STOP... ");
			i2c_send_stop();
			reply("sent");
			break;
		default:
			reply("Unrecognised command.");
			break;
		}
	}

	return 0;
}
